<?php

namespace App\Http\Controllers\Institution;

use App\Http\Controllers\Controller;
use App\Http\Requests\InstitutionRequest;
use App\Http\Requests\InstitutionStoreRequest;
use App\Http\Requests\InstitutionUpdateRequest;
use App\Models\Institution;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;

class InstitutionController extends Controller
{
    //
    public function index()
    {
        $institutions = Institution::all();
        $idInstitutionTable = 'institution-table';

        $headers = [
            '#',
            'Nama Lembaga',
            'ID Lembaga',
            'E-mail'
        ];
        $rows = [
            // [
            //     1,
            //     'Nurul Hayah II',
            //     '10632',
            //     'info@nurulhayah2.com'
            // ]
        ];

        $no = 1;
        foreach ($institutions as $institution) {
            $rows[] = [
                $no++,
                $institution->name,
                $institution->institution,
                $institution->email,
                $institution->id,
            ];
        }


        return view('pages.admin.institution.index', compact('idInstitutionTable', 'headers', 'rows'));
    }

    public function edit($id)
    {
        $institution = Institution::with('user')->where('id', $id)->first();

        return response()->json($institution);
    }

    public function store(InstitutionRequest $request)
    {
        $notif = ['status' => 'notif-success', 'success' => true, 'description' => 'Berhasil menambahkan data!'];
        $institutionData = $request->only([
            'name',
            'bank_account_number',
            'bank_account_holder_name',
            'province_hide',
            'regency_hide',
            'district_hide',
            'sub_district_hide',
            'email',
            'institution',
        ]);

        $userOrAdminData = $request->only([
            'admin_name',
            'admin_username',
            'admin_phone',
            'admin_email',
            'admin_pass'
        ]);

        // validation bank account number and nis number
        if (!$this->bankAccountIsValidate($institutionData['bank_account_number'])) {
            $notif['status'] = 'notif-error';
            $notif['success'] = false;
            $notif['description'] = 'Nomer Rekening Tidak Ada di BSN!';
            return response()->json($notif);
        }

        $originalName = $institutionData['name'];

        if ($request->file('logo_file')) {
            $institutionData['logo'] = $request->file('logo_file')->store('logo', 'public');
        } else {
            $institutionData['logo'] = null;
        }



        try {
            DB::beginTransaction();

            $user = User::create([
                'name' => $userOrAdminData['admin_name'],
                'username' => $userOrAdminData['admin_username'],
                'phone' => $userOrAdminData['admin_phone'],
                'email' => $userOrAdminData['admin_email'],
                'password' => bcrypt($userOrAdminData['admin_pass']),
                'status' => '',
                'role' => 'INSTITUTION',
            ]);


            $notif = insertBasicTableFormatted(new Institution, [
                'user_id' => $user->id,
                'name' => $institutionData['name'],
                'bank_account_number' => $institutionData['bank_account_number'],
                'bank_account_holder_name' => $institutionData['bank_account_holder_name'],
                'province' => $institutionData['province_hide'],
                'regency' => $institutionData['regency_hide'],
                'district' => $institutionData['district_hide'],
                'sub_district' => $institutionData['sub_district_hide'],
                'email' => $institutionData['email'],
                'institution' => $institutionData['institution'],
                'logo' => $institutionData['logo']
            ], $notif, $originalName);

            if (!$notif['success']) {
                unlink(public_path('storage/' . $institutionData['logo']));
            }


            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;

            $notif['status'] = 'notif-error';
            $notif['description'] = 'Gagal menambahkan data, Silahkan coba lagi!.';
            unlink(public_path('storage/' . $institutionData['logo']));
        }

        return response()->json($notif);
    }

    public function update(InstitutionRequest $request, $id)
    {
        $institution = Institution::findOrFail($id);

        $user = User::findOrFail($institution->user_id);


        $institutionData = $request->only([
            'name',
            'bank_account_number',
            'bank_account_holder_name',
            'province_hide',
            'regency_hide',
            'district_hide',
            'sub_district_hide',
            'email',
            'institution',
            'old_logo_file',
        ]);

        $userOrAdminData = $request->only([
            'admin_name',
            'admin_username',
            'admin_phone',
            'admin_email',
            'admin_pass'
        ]);

        $originalName = $institutionData['name'];
        $notif = ['status' => 'notif-success', 'success' => true, 'description' => 'Berhasil mengupdate data!'];

        if ($request->file('logo_file')) {

            if ($institutionData['old_logo_file']) {
                Storage::delete($institutionData['old_logo_file']);
            }

            $institutionData['logo'] = $request->file('logo_file')->store('logo', 'public');
        } else {
            $institutionData['logo'] = null;
        }

        try {
            DB::beginTransaction();

            $user->update([
                'name' => $userOrAdminData['admin_name'],
                'username' => $userOrAdminData['admin_username'],
                'phone' => $userOrAdminData['admin_phone'],
                'email' => $userOrAdminData['admin_email'],
                'password' => bcrypt($userOrAdminData['admin_pass']),
                'status' => '',
                'role' => 'INSTITUTION',
            ]);


            $notif = updateBasicTableFormatted(new Institution, $id, [
                'user_id' => $user->id,
                'name' => $institutionData['name'],
                'bank_account_number' => $institutionData['bank_account_number'],
                'bank_account_holder_name' => $institutionData['bank_account_holder_name'],
                'province' => $institutionData['province_hide'],
                'regency' => $institutionData['regency_hide'],
                'district' => $institutionData['district_hide'],
                'sub_district' => $institutionData['sub_district_hide'],
                'email' => $institutionData['email'],
                'institution' => $institutionData['institution'],
                'logo' => $institutionData['logo']
            ], $notif, $originalName);

            if (!$notif['success']) {
                unlink(public_path('storage/' . $institutionData['logo']));
            }


            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;

            $notif['status'] = 'notif-error';
            $notif['description'] = 'Gagal menambahkan data, Silahkan coba lagi!.';
            unlink(public_path('storage/' . $institutionData['logo']));
        }

        session()->forget('http_method');

        return response()->json($notif);
    }

    public function destroy(Request $request)
    {
        $notif = ['status' => 'notif-success', 'success' => true, 'description' => 'Berhasil menghapus data!'];
        $ids = empty($request->ids) || !isset($request->ids) ? $request->id_confirm : $request->ids;

        try {
            if ($ids[0] == 'on') {
                array_shift($ids);
            }
            if (is_array($ids) && count($ids) > 0) {
                // Dapatkan ID institusi yang akan dihapus
                $institutions = Institution::whereIn('id', $ids)->get();

                // Loop melalui institusi-institusi tersebut
                foreach ($institutions as $institution) {
                    // Dapatkan ID pengguna yang terkait
                    $userId = $institution->user_id;

                    // Hapus baris institusi
                    $institution->delete();

                    // Hapus baris pengguna yang terkait
                    User::where('id', $userId)->delete();
                }
            } else {
                $institution = Institution::findOrFail($ids[0]);
                $userId = $institution->user_id;

                // Hapus baris institusi
                $institution->delete();

                // Hapus baris pengguna yang terkait
                User::where('id', $userId)->delete();
            }
        } catch (\Throwable $th) {
            $notif['status'] = 'notif-error';
            $notif['description'] = 'Gagal menghapus data, Silahkan coba lagi!.';
        }

        return response()->json($notif);
    }


    public function bankAccountIsValidate($bankAccountNumber)
    {
        // Lakukan permintaan HTTP ke URL yang diberikan
        $response = Http::withoutVerifying()->get("https://ib.bsn.co.id/ApiSantri/index/{$bankAccountNumber}");

        $trimmedResponse = trim($response->body());

        // Periksa apakah body respons sama dengan string yang diharapkan setelah di-trim
        if (strpos($trimmedResponse, '{"status":"error","message":"No Rekening Invalid."}') !== false) {
            return false;
        }

        return true;
    }
}
